/******************************************************************************
*                                                  
*  (c) copyright Freescale Semiconductor 2008
*  ALL RIGHTS RESERVED
*
*  File Name: Bootloader_S08.C
*                                                                          
*  Purpose: This file is for a USB Mass-Storage Device bootloader.  This file 
*           has the main routine for the bootloader
*                                                                          
*  Assembler:  Codewarrior for Microcontrollers V6.2
*                                            
*  Version:  1.6
*                                                                          
*                                                                          
*  Author: Derek Snell                             
*                                                                                       
*  Location: Indianapolis, IN. USA                                            
*                                                                                  
* UPDATED HISTORY:
*
* REV   YYYY.MM.DD  AUTHOR        DESCRIPTION OF CHANGE
* ---   ----------  ------        --------------------- 
* 1.6   2009.11.18  Derek Snell   Initialized vCBWBuf_flag to 0.
* 1.5   2009.01.05  Derek Snell   Ported from V1 JM128 to S08 JM60
* 1.4   2008.11.24  Derek Snell   Added EP0 stall after GetMAXLUN request to signify only one LUN
* 1.3   2008.09.17  Derek Snell   After file written, force re-enumeration to update file displayed to host
* 1.2   2008.08.26  Derek Snell   Fix MaxLUN response so only 1 drive enumerates
*                   & Jose Ruiz   Added proper Endpoint 0 stalls
*                                 Stopped accepting write data from host after successful S19 transfer
* 1.1   2008.08.14  Derek Snell   Modified to improve enumeration delay
* 1.0   2008.06.10  Derek Snell   Initial version
* 
*
******************************************************************************/                                                                        
/* Freescale  is  not  obligated  to  provide  any  support, upgrades or new */
/* releases  of  the Software. Freescale may make changes to the Software at */
/* any time, without any obligation to notify or provide updated versions of */
/* the  Software  to you. Freescale expressly disclaims any warranty for the */
/* Software.  The  Software is provided as is, without warranty of any kind, */
/* either  express  or  implied,  including, without limitation, the implied */
/* warranties  of  merchantability,  fitness  for  a  particular purpose, or */
/* non-infringement.  You  assume  the entire risk arising out of the use or */
/* performance of the Software, or any systems you design using the software */
/* (if  any).  Nothing  may  be construed as a warranty or representation by */
/* Freescale  that  the  Software  or  any derivative work developed with or */
/* incorporating  the  Software  will  be  free  from  infringement  of  the */
/* intellectual property rights of third parties. In no event will Freescale */
/* be  liable,  whether in contract, tort, or otherwise, for any incidental, */
/* special,  indirect, consequential or punitive damages, including, but not */
/* limited  to,  damages  for  any loss of use, loss of time, inconvenience, */
/* commercial loss, or lost profits, savings, or revenues to the full extent */
/* such  may be disclaimed by law. The Software is not fault tolerant and is */
/* not  designed,  manufactured  or  intended by Freescale for incorporation */
/* into  products intended for use or resale in on-line control equipment in */
/* hazardous, dangerous to life or potentially life-threatening environments */
/* requiring  fail-safe  performance,  such  as  in the operation of nuclear */
/* facilities,  aircraft  navigation  or  communication systems, air traffic */
/* control,  direct  life  support machines or weapons systems, in which the */
/* failure  of  products  could  lead  directly to death, personal injury or */
/* severe  physical  or  environmental  damage  (High  Risk Activities). You */
/* specifically  represent and warrant that you will not use the Software or */
/* any  derivative  work of the Software for High Risk Activities.           */
/* Freescale  and the Freescale logos are registered trademarks of Freescale */
/* Semiconductor Inc.                                                        */ 
/*****************************************************************************/



#include "derivative.h" 
#include "hidef.h"
#include "bootloader_S08.h"
#include "SCSI_Process.h"

/************************************************************************************
*************************************************************************************
* Global Variables
*************************************************************************************
************************************************************************************/
tICPSTR _ICP_VAR;
#define ICP_VAR               _ICP_VAR.Byte
#define ICP_AddrPend          _ICP_VAR.Bits.AddrPend


// USB Buffer Descriptor Tables and buffers
volatile tBDT EP0Tx                         @(USB_BDT_START);
volatile tBDT EP0Rx                         @(USB_BDT_START +   BDT_SIZE);
volatile tBDT EP1Tx                         @(USB_BDT_START + 2*BDT_SIZE);
volatile tBDT EP2Rx                         @(USB_BDT_START + 3*BDT_SIZE);
volatile byte ICP_IN_Data[mEP0_BUFF_SIZE]   @(USB_BUFFER_START);
volatile tUSB_Setup Setup_Pkt               @(USB_BUFFER_START +   mEP0_BUFF_SIZE);
volatile byte EP1Tx_Data[cEP1_BUFF_SIZE]    @(USB_BUFFER_START + 2*mEP0_BUFF_SIZE);
volatile byte EP2Rx_Data[cEP2_BUFF_SIZE]    @(USB_BUFFER_START + 2*mEP0_BUFF_SIZE + cEP1_BUFF_SIZE);

// Variables used with USB buffers and USB controller
byte *pEP0IN_Data;
byte vEP0IN_DataCnt;
word vEP1Idx;
word vEP2Idx;
byte vEP1Data[MSD_BUFFER_SIZE];
byte vEP2Data[MSD_BUFFER_SIZE];  
word vEP1_Cnt;
byte vEP0RxData;          // data0/1 toggle
byte vEP0TxData;          // data0/1 toggle
byte vEP1TxData;          // data0/1 toggle
byte vEP2RxData;          // data0/1 toggle

// USB Stack Variables
byte ICP_USB_State;
byte vCBWBuf_flag;

// Variables for Flash update
volatile unsigned char FLASHPGM[59];
byte result;

// Flash Protection and Security
const byte NVPROT_INIT @0x0000FFBD = 0xEA;    // 0xEC00-0xFFFF are protected 
const byte NVOPT_INIT  @0x0000FFBF = 0x42;    // no vector redirect, flash unsecure


// Bootloader Variables
unsigned char BootloaderStatus;
unsigned char FlashErased;
unsigned char ReEnumerateNow;

/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/

extern byte               vSCSIState;         // State of SCSI State Machine
extern word               vEP1_Cnt;
extern dword	            v_LBA;              // BLock address
extern byte               S19FileDone;
extern byte               BlockWriteDone;
extern byte               vCSWResult;         // CSW result
extern byte               vCBW_Buf[31];
extern byte               vCSWResult; 
extern byte               MultiLBAToken;


/************************************************************************************
*************************************************************************************
* Public funcitons
*************************************************************************************
************************************************************************************/
void SCSI_Init(void);
void ParseS19(void);
void GetUSBOutFileData(void);
void Send_CSW(void);
void SCSI_Process(void);



/********************************************************************
*********************************************************************
*     Flash subroutines
*********************************************************************
********************************************************************/

const unsigned char FLASHPGM_ROM[59] = {  
0x87,0xC6,0x18,0x25,0xA5,0x10,0x27,0x08,0xC6,0x18,0x25,0xAA,0x10,0xC7,0x18,0x25,
0x9E,0xE6,0x01,0xF7,0xA6,0x20,0xC7,0x18,0x26,0x45,0x18,0x25,0xF6,0xAA,0x80,0xF7,
0x9D,0x9D,0x9D,0x9D,0x45,0x18,0x25,0xF6,0xF7,0xF6,0xA5,0x30,0x27,0x04,0xA6,0xFF,
0x20,0x07,0xC6,0x18,0x25,0xA5,0x40,0x27,0xF9,0x8A,0x81};

/*  The opcode above represents this set of instructions  
    if (FSTAT&0x10){                     //Check to see if FACCERR is set
        FSTAT = FSTAT | 0x10;            //write a 1 to FACCERR to clear
    }
    (*((volatile unsigned char *)(Address))) = data;  //write to somewhere in flash
    FSTAT = 0x80;                        //Put FCBEF at 1.
    _asm NOP;                            //Wait 4 cycles
    _asm NOP;
    _asm NOP;
    _asm NOP;
    if (FSTAT&0x30){                     //check to see if FACCERR or FVIOL are set
    return 0xFF;                         //if so, error.
    }
    while ((FSTAT&0x40)==0){             //else wait for command to complete
        ;
    }*/


/*********************************************************
* Name: FLASHPGM_Copy
*
* Desc: Copy flash routine to RAM   
*             
**********************************************************/
void FLASHPGM_Copy(void) 
{
  unsigned char *pSrc;
  unsigned char *pDst;
  unsigned char i;
  
  pSrc = (unsigned char *)FLASHPGM_ROM;
  pDst = (unsigned char *)FLASHPGM;
  
  
  for(i=59;i>0;i--) 
  {
     *pDst=*pSrc;
     pSrc++;
     pDst++;
  }
  
}

/*********************************************************
* Name: Flash_Prog
*
* Desc: Flash programming function   
*             
**********************************************************/
#pragma MESSAGE DISABLE C1805   // Disable this warning for this function
unsigned char Flash_Prog(unsigned int dst_addr, unsigned int data_addr, unsigned char length) 
{
    unsigned char data,temp;
    
    
    for(;length>0;length--) 
    {
      data=*(unsigned char *)data_addr;
      
      temp = Program_Byte(dst_addr,data);
      
      if(gFlashError == temp){
        return gFlashError;
      }
      
      dst_addr++;
      data_addr++;
    }
    
    return gFlashSuccess;
}
#pragma MESSAGE WARNING C1805   // Re-Enable this warning



/*********************************************************
* Name: Flash_verify
*
* Desc: Flash verification
*             
**********************************************************/
/*unsigned char Flash_Verify(unsigned int dst_addr, unsigned int data_addr, unsigned char length) 
{
    unsigned char *flash, *ram;
    unsigned char status;
    
    
    flash = (unsigned char *)dst_addr;
	  ram = (unsigned char *)data_addr;
    
    do
	 {
		  if(*flash++ == *ram++)			        // compare two values
		{
		   status = gFlashSuccess;						// if same, continue to compare 
		}	
		else
		{
		    status = gFlashError;							// if different, break
		    break;
		 }
		 
	}	 
	while(--length);		
    
   return status;
}*/




/*********************************************************
* Name: Flash_Erase
*
* Desc: Flash earse function   
*             
**********************************************************/
/*unsigned char Flash_Erase(unsigned int addr) 
{
  unsigned char temp;
  
   
  temp=Page_Erase(addr);
  
  
  if(gFlashError == temp)
    return gFlashError;
  
  return gFlashSuccess;
}*/


/*********************************************************
* Name: CheckAddressValid
*
* Desc: Checks if Address of S-Record is valid for device
*
* Parameter: Address
*
* Return: unsigned char, TRUE or FALSE if valid
*             
**********************************************************/
unsigned char CheckAddressValid(dword Address) {

    if((Address >= MIN_FLASH1_ADDRESS) && (Address <= MAX_FLASH1_ADDRESS))
        return TRUE;
    else if((Address >= MIN_FLASH2_ADDRESS) && (Address <= MAX_FLASH2_ADDRESS)) 
        return TRUE;
    else
        return FALSE;
}

/*********************************************************
* Name: EraseFlash
*
* Desc: Erases all unprotected pages of flash
*
* Parameter: None
*
* Return: None
*             
**********************************************************/
#pragma MESSAGE DISABLE C1805   // Disable this warning for this function
void EraseFlash(void) {
    
    word addr;
    unsigned char temp;
    
    // erase each page of flash
    for(addr=MIN_FLASH2_ADDRESS;addr<=MAX_FLASH2_ADDRESS;addr+=FLASH_PAGE_SIZE) {
        if(addr < FLASH_PROTECTED_ADDRESS) {
            temp = Page_Erase(addr);
            if(gFlashError == temp) {
                // Error Erasing Flash
                BootloaderStatus = BootloaderFlashError;
                return;
            }
        } else
            break;
    }

    for(addr=MIN_FLASH1_ADDRESS;addr>=MIN_FLASH1_ADDRESS;addr+=FLASH_PAGE_SIZE) {
        if(addr < FLASH_PROTECTED_ADDRESS) {
            temp = Page_Erase(addr);
            if(gFlashError == temp) {
                // Error Erasing Flash
                BootloaderStatus = BootloaderFlashError;
                return;
            }
        } else
            break;
    }
}
#pragma MESSAGE WARNING C1805   // Re-Enable this warning


/********************************************************************
*********************************************************************
*       USB Device Descriptor
*********************************************************************
********************************************************************/
const byte Device_Descriptor[18]= 
{
	0x12,		      //blength
	0x01,		      //bDescriptor
	0x00,0x02,	  //bcdUSB ver R=2.00 
	0x00,		      //bDeviceClass
	0x00,		      //bDeviceSubClass			
	0x00,		      //bDeviceProtocol			
	0x10,		      //bMaxPacketSize0
	0xA2,0x15,	  //idVendor - 0x15A2(freescale Vendor ID)
	0x3F,0x00,	  //idProduct - 0x003F
	0x00,0x01,	  //bcdDevice - Version 1.00
	0x01,		      //iManufacturer - Index to string descriptor
	0x02,		      //iProduct - Index to string descriptor
	0x00,		      //iSerialNumber - Index to string descriptor
	0x01	       	//bNumConfigurations - # of config. at current speed,
};

/********************************************************************
*********************************************************************
*       USB Configuration Descriptor
*********************************************************************
********************************************************************/
const byte Configuration_Descriptor[32]= 
{
	0x09,		      //blength
	0x02,		      //bDescriptor
	0x20,0x00,	  //wTotalLength - # of bytes including interface and endpoint descpt.
	0x01,		      //bNumInterfaces - at least 1 data interface
	0x01,		      //bConfigurationValue - 
	0x00,		      //iConfiguration - index to string descriptor	
	0x80,		      //bmAttributes - 0x?0	bit 7-bus powered
				        //						        bit 6-self powered
				        //						        bit 5-remote wakeup
				        //						        bit 4-0-reserved
	0x64,		      //bMaxPower - 200mA
	
// ======================================================
// 		Standard Interface Descriptor
// ======================================================

	0x09,		      //blength
	0x04,		      //bDescriptorType - Interface descriptor
	0x00,		      //bInterfaceNumber - Zero based value identifying the index of the config.
	0x00,		      //bAlternateSetting;
	0x02,		      //bNumEndpoints - 2 endpoints
	0x08,		      //bInterfaceClass - mass storage 
	0x06,		      //bInterfaceSubClass - SCSI Transparent command Set
	0x50,		      //bInterfaceProtocol - Bulk-Only transport
	0x00,		      //iInterface - Index to String descriptor
	
	
// ======================================================
// 		Standard Endpoint Descriptor
//    	Bulk IN Endpoint
// ======================================================
	0x07,		      //bLength;
	0x05,		      //bDescriptorType - Endpoint descriptor
	0x81,		      //bEndpointAddress - bit 3..0 endpoint number
				        // 					         bit 6..4 Reserved reset to zero
				        //					         bit 7	0-OUT, 1-IN									
	0x02,		      //bmAttributes _ Bulk endpoint
	0x40,0x00,	  //wMaxPacketSize - 64 byte packet size
	0x00,		      //bInterval - do not apply to BULK endpoint
	
// ======================================================
// 		Standard Endpoint Descriptor
//    	Bulk OUT Endpoint
// ======================================================
	0x07,		      //bLength;
	0x05,		      //bDescriptorType - Endpoint descriptor
	0x02,		      //bEndpointAddress - bit 3..0 endpoint number
				        // 					         bit 6..4 Reserved reset to zero
				        //					         bit 7	0-OUT, 1-IN									
	0x02,		      //bmAttributes _ Bulk endpoint
	0x40,0x00,	  //wMaxPacketSize - 64 byte packet size
	0x00,		      //bInterval - do not apply to BULK endpoint	
 
};



// ======================================================
//
// 		String Descriptor Zero
//
// ======================================================
const byte String_Descriptor0[4] = {
	0x04,		      //bLength;
	0x03,		      //bDescriptorType - STRING descriptor
	0x09,0x04,	  //wLANDID0 - English (American)
};

// ======================================================
//
// 		String Descriptor one
//
// ======================================================
const byte String_Descriptor1[] = {
	0x14,			    //bLength; 11 bytes
	0x03,		      //bDescriptorType - STRING descriptor
	'F',0x00,	    // "F"
	'r',0x00,	    // "r"
	'e',0x00,	    // "e"
	'e',0x00,	    // "e"
	's',0x00,	    // "s"
	'c',0x00,	    // "c"
	'a',0x00,	    // "a"
	'l',0x00,	    // "l"
	'e',0x00,	    // "e"
};	 


// ======================================================
//
// 		String Descriptor two
//
// ======================================================
const byte String_Descriptor2[] = {
	0x0A,			    //bLength;
	0x03,		      //bDescriptorType - STRING descriptor
	'J',0x00,	    // JM60
	'M',0x00,	
	'6',0x00,	
	'0',0x00,
};	 


/********************************************************************
*********************************************************************
*     USB subroutines
*********************************************************************
********************************************************************/

/*********************************************************
* Name: USB_InitBulk
*
* Desc: Initialize EP1 and EP2 as Bulk only endpoint
*       EP1- IN  EP2-OUT
* 
* Parameter: None
*
* Return: None
*            
**********************************************************/

void USB_InitBulk(void) 
{
  EPCTL1=0x05;                          // endpoint 1 for IN only
  EPCTL2=0x09;                          // endpoint 2 for OUT only
  
  EP1Tx.Stat._byte= mUDATA1;
  EP1Tx.Cnt = 0x00;
  EP1Tx.Addr = cEP1INBuffAddr;          // EP1 IN buffer;
  vEP1TxData = mUDATA1;
  
  EP2Rx.Stat._byte = mUDATA0;
  EP2Rx.Cnt = cEP2_BUFF_SIZE;
  EP2Rx.Addr = cEP2OUTBuffAddr;         // EP2 OUT buffer;
  vEP2RxData = mUDATA0;
  
  vEP2Idx = 0;
  
}


/*********************************************************
* Name: Init_USB
*
* Desc: Initialize USB module, enable USB module
*
* Parameter: None
*
* Return: None
*             
**********************************************************/

void Init_USB(void)
{
  
  USBCTL0_USBRESET = 1;                     // reset USB modult

  //Below is initialize the BDT
  EP0Tx.Stat._byte = mUDATA1;     // endpoint 0 IN buffer initialization
  EP0Tx.Cnt = 0x00;                      
  EP0Tx.Addr = cEP0INBuffAddr;                     
  
  EP0Rx.Stat._byte = mUDATA0;     // endpoint 0 OUT initialization
  EP0Rx.Cnt = mEP0_BUFF_SIZE;              
  EP0Rx.Addr = cEP0OUTBuffAddr;   // (EP0 OUT buffer-0x1860) >> 2;    

  INTENB = 0x00;                      // disable USB interrupts
  ERRENB = 0x00;                      // disable all USB error interrupt sources
  ERRSTAT = 0xFF;                     // clear USB error flag
  INTSTAT = 0xBF;                     // clear USB interrupt

  EPCTL0 = 0x0D;                      // enable endpoint 0    
  
  ICP_USB_State = cPOWER;             // Device in power state

  USBCTL0=0x45;                       // enables USB on-chip regulator and pulls up USBDP

  CTL_USBEN = 1;                      // enable USB module
  
}
  


/*********************************************************
* Name: EP0_Load
*
* Desc: Load EP0 data to EP0 IN buffer
*
* Parameter: None
*
* Return: None
*             
**********************************************************/

void EP0_Load(void)
{    
    byte Byte_To_Send;
    byte *pDst;
    
    if(vEP0IN_DataCnt < mEP0_BUFF_SIZE)
        Byte_To_Send = vEP0IN_DataCnt;
    else
        Byte_To_Send = mEP0_BUFF_SIZE;
    
    pDst=(byte *)&ICP_IN_Data;
    
    EP0Tx.Cnt = Byte_To_Send;
    vEP0IN_DataCnt = vEP0IN_DataCnt - Byte_To_Send;
    
    while(Byte_To_Send)
    {
       *pDst = *pEP0IN_Data;
       pDst++;
       pEP0IN_Data++;
       Byte_To_Send--;
    }
      
    EP0Tx.Stat._byte = vEP0TxData;
    vEP0TxData ^=0x40;
}     

/*********************************************************
* Name: EP0_Stall
*
* Desc: Stalls Endpoint 0
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/
void EP0_Stall(void)
{

   EPCTL0_EPSTALL = 1;                // stall EP0
   EP0Rx.Stat._byte = mUDATA0;        // endpoint 0 OUT initialization
   EP0Rx.Cnt = mEP0_BUFF_SIZE;              
   
}



/*********************************************************
* Name: EP1_Load
*
* Desc: Load data to Endpoint 1 buffer
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/
void EP1_Load(void)
{
    byte i,counter;
    byte *pBuffer;
     
    pBuffer=(byte *)&EP1Tx_Data;
    
    if(vEP1_Cnt > cEP1_BUFF_SIZE)
      counter = cEP1_BUFF_SIZE;
    else
      counter = (byte)vEP1_Cnt;
    
    for(i=0;i<counter;i++,vEP1Idx++){
      pBuffer[i]=vEP1Data[vEP1Idx];
    }
       
    
    EP1Tx.Cnt = counter;
    EP1Tx.Stat._byte= vEP1TxData;
    vEP1TxData ^=0x40;  

    vEP1_Cnt = (word)(vEP1_Cnt - counter);
    
}


void ICP_USB_Wait(void)
{
            
    CTL_ODDRST = 1; 
    
    vEP0RxData = mUDATA0;
    vEP0TxData = mUDATA0;
    
    CTL_ODDRST = 0;
      
    EP0Rx.Stat._byte = mUDATA0;
	  EP0Rx.Cnt = mEP0_BUFF_SIZE;
	  
    EP0Tx.Stat._byte = mMCU;
	  EP0Tx.Cnt = mEP0_BUFF_SIZE;
   
    EP1Tx.Stat._byte= mUDATA0;
    EP1Tx.Cnt = cEP1_BUFF_SIZE;
    vEP1TxData = mUDATA0;
  
    EP2Rx.Stat._byte = mUDATA0;
    EP2Rx.Cnt  = cEP2_BUFF_SIZE;
    vEP2RxData = mUDATA0;
    
    CTL_TSUSPEND = 0; 
    
    ICP_USB_State = mICP_WAIT;              
   
}

/*********************************************************
* Name: USB_CLsReq_Handler
*
* Desc: Handle USB class request
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/
void USB_ClsReq_Handler(void)
{
  
   if(mGET_MAXLUN == Setup_Pkt.bRequest)
   {
      EP0_Stall();
   } 
   else
   {
      ICP_USB_Wait();
   }
}
   



/*********************************************************
* Name: ICP_Setup_Handler
*
* Desc: Handle USB SETUP token
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/

void ICP_Setup_Handler(void)
{
   byte length;
   
   vEP0IN_DataCnt= 0;
   ICP_USB_State = mICP_WAIT;
   
   if(0x00 == (Setup_Pkt.bmRequestType & 0x60)) {
      
      switch(Setup_Pkt.bRequest) 
      {
     
        case mGET_STATUS:
          pEP0IN_Data = (byte*)&Device_Descriptor+4; 			
          vEP0IN_DataCnt = 0x02;   
          break;
      
        case mSET_ADDR:
          ICP_AddrPend = 1;
          break;
      
        case mGET_DESC:
        
            switch(Setup_Pkt.wValue_h) 
            {
              case mDESC_DEV:
                pEP0IN_Data = (byte*)&Device_Descriptor;
                vEP0IN_DataCnt = sizeof(Device_Descriptor);
                break;
          
              case mDESC_CFG:
                pEP0IN_Data = (byte*)&Configuration_Descriptor;
                vEP0IN_DataCnt = sizeof(Configuration_Descriptor);
                break;
        
              case mDESC_STR:
                      switch (Setup_Pkt.wValue_l) {
                          case 0:
                              pEP0IN_Data = (byte*)String_Descriptor0;
                              break;
                          case 1:
                              pEP0IN_Data = (byte*)String_Descriptor1;
                              break;
                          case 2:
                              pEP0IN_Data = (byte*)String_Descriptor2;
                              break;
                      }
                      vEP0IN_DataCnt = *pEP0IN_Data;
                      break;
          
              default: 
                EP0_Stall();
                break;  
            }
        
            break;

        case mSET_CFG:
              if(Setup_Pkt.wValue_h+Setup_Pkt.wValue_l) // value is not zero
              {
                  USB_InitBulk();                         // init EP1 and EP2
            
              } 
              else
                  ICP_AddrPend = 1;
              
              break;
      
        default:                         // not support
            EP0_Stall();
            break;
          
      } 
   } else if(0x20 == (Setup_Pkt.bmRequestType & 0x60)) {
        USB_ClsReq_Handler();       
   } else                               // not supported
        EP0_Stall();



   if(Setup_Pkt.bmRequestType &0x80)              // Device to Host
    {
       
       length = Setup_Pkt.wLength_l;
              
  		 if(length < vEP0IN_DataCnt)
         vEP0IN_DataCnt = length;
  			
       vEP0TxData = mUDATA1; 
       EP0_Load();
       ICP_USB_State   = mICP_TX; 
      
    }
    else    
    {
       ICP_USB_State   = mICP_RX;
       EP0Tx.Cnt = 0x00;                              // return zero IN data
       EP0Tx.Stat._byte = mUDATA1;  
    }
  
         
 	 vEP0RxData ^= 0x40;
	 EP0Rx.Cnt = mEP0_BUFF_SIZE;
         
   EP0Rx.Stat._byte = vEP0RxData; 
   CTL_TSUSPEND = 0;
 
}

/*********************************************************
* Name: ICP_In_Handler
*
* Desc: Handle USB EP0 IN Token
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/

void ICP_In_Handler(void)
{
   
    if(ICP_AddrPend ==1) {
        ADDR = Setup_Pkt.wValue_l;
        ICP_AddrPend =0;
    }
           
    if(ICP_USB_State == mICP_TX)
    {
      EP0_Load();
    } 
    
    else
      ICP_USB_Wait(); 
}


/*********************************************************
* Name: ICP_Reset_Handler
*
* Desc: Handle USB reset command
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/

void ICP_Reset_Handler(void)
{
 
   INTSTAT = 0xBF;              // clear all flag
   INTENB = 0x00;
   ADDR = 0x00;
   ICP_USB_Wait();   
    
}

/*********************************************************
* Name: USB_EP1_IN_Handler
*
* Desc: Handle USB Endpoint1 IN token
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/
void USB_EP1_IN_Handler(void)
{
   
   if( ICP_USB_State == cCSW) 
   {
      Send_CSW();
   } 
   
   else if(ICP_USB_State == cEP1Tx)
   {
     
     EP1_Load();
     if(vEP1_Cnt == 0)
    {
      ICP_USB_State=mICP_WAIT; 
    } 
    
   }
   
   else
   {
      EP1Tx.Cnt = 0;
      EP1Tx.Stat._byte= mMCU;  
   }
    
}


/*********************************************************
* Name: USB_EP2_OUT_Handler
*
* Desc: Handle USB Endpoint2 OUT token
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/ 
void USB_EP2_OUT_Handler(void)
{
    byte i;
    byte *pCBW_Pkt;
    
    pCBW_Pkt=(byte *)&EP2Rx_Data;
      

    if(EP2Rx.Cnt ==31)        // CBW should be 31 bytes
    {
      ICP_USB_State = cCBW;
      vCBWBuf_flag=1;   
      
      for(i=0;i<EP2Rx.Cnt;i++)
      {
          vCBW_Buf[i]= pCBW_Pkt[i];
          
      }
             
      vEP2RxData ^=0x40;
      EP2Rx.Stat._byte=vEP2RxData;
      EP2Rx.Cnt  = cEP2_BUFF_SIZE;
    }
      
    else
    {
         
      for(i=0;i<EP2Rx.Cnt;i++,vEP2Idx++)
      {
         vEP2Data[vEP2Idx]= pCBW_Pkt[i];
      }
         
      if(vEP2Idx == 512)
      {
        ICP_USB_State = cEP2Rx;
        vEP2RxData ^=0x40;
        EP2Rx.Cnt  = cEP2_BUFF_SIZE;
        EP2Rx.Stat._byte=vEP2RxData;
      } 
      else
      {
          
        vEP2RxData ^=0x40;
        EP2Rx.Cnt  = cEP2_BUFF_SIZE;
        EP2Rx.Stat._byte=vEP2RxData;
      }  
        
    }
}

/*********************************************************
* Name: USB_Stall_Handler
*
* Desc: Handles Endpoint 0 Stall
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/ 
void USB_Stall_Handler(void) 
{
    
    if(EPCTL0_EPSTALL == 1)
        EPCTL0_EPSTALL = 0;
    
    INTSTAT_STALLF = 1;         // clear stall flag
}


/*********************************************************
* Name: PollUSB
*
* Desc: Polls USB flags instead of using interrupts
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/ 
void PollUSB(void) {

    byte stat;

    if(INTSTAT_USBRSTF) 
    {
        ICP_Reset_Handler();
    }
    
    if(INTSTAT_STALLF)
        USB_Stall_Handler();
    
    if(INTSTAT_TOKDNEF)
    {
        stat=(byte)(STAT &0xF8);
         
        // OUT or SETUP token
        if(stat == mEP0_OUT)
        {
          // SETUP token
          if(EP0Rx.Stat.RecPid.PID == mSETUP_TOKEN)
          {
            ICP_Setup_Handler();
          }  
          
          // OUT token
          else                                              
            ICP_USB_Wait();
        }
        
        else if(stat == mEP0_IN)
        {
          // IN token                                 
           ICP_In_Handler();
        }
        else if (stat==mEP1_IN){
           USB_EP1_IN_Handler();
        }
        else if (stat==mEP2_OUT) {
           USB_EP2_OUT_Handler();
        }

        // Clear token flag  
        INTSTAT_TOKDNEF = 1; 
        MultiLBAToken = 0;  
    
    }
  
}

void Bootloader_Main(void) 
{
  word i;
  byte j;

  DisableInterrupts;
  
  //clear ram
  ICP_VAR = 0x00;
  vEP0IN_DataCnt = 0x00;
  vEP0RxData = 0x00;          
  vEP0TxData = 0x00;
  vEP1TxData = 0x00;
  vEP2RxData = 0x00;
  vCBWBuf_flag = 0x00;          

	BootloaderStatus = BootloaderReady;
	S19FileDone = TRUE;
	FlashErased = FALSE;
  ReEnumerateNow = NO;

  FCDIV=0x4F;               // for bus clock=24MHz
                            
	FLASHPGM_Copy();

	SCSI_Init();
	Init_USB();

   for(;;)
   {
      PollUSB(); 
      
      if(vCBWBuf_flag==1)
      {
          SCSI_Process();
          vCBWBuf_flag=0;

          // Force re-enumeration to update file displayed to host
          // after file has been written
          if(ReEnumerateNow == YES) {
              // Force Re-enumeration
              CTL_USBEN = 0;                    // Disable USB controller
              USBCTL0 = 0;                      // Disable PHY and Pull-Up
          
              for(i=0;i<25000;i++){               // delay
                  for(j=0;j<255;j++) {
                      asm(nop);  
                      asm(nop);  
                      asm(nop);  
                  }
              }
          
              
              Init_USB();                       // Enumerate again
              ReEnumerateNow = DONE;
            
          }
          
      }
      if(!S19FileDone && (BootloaderStatus == BootloaderReady)){
          BootloaderStatus = BootloaderStarted;
          ParseS19();
          while(!BlockWriteDone)                // Finish USB transfer
              GetUSBOutFileData();              
          vCSWResult = kCSWPass;
          Send_CSW();                       // finish current USB transfer
      }    
    
   }// for end

}






